/*
 * Top-level definitions for pyregfi to be processed by Michael Cohen's 
 * automated Python bindings generator.
 *
 * Copyright (C) 2010 Michael I. Cohen
 * Copyright (C) 2010 Timothy D. Morgan
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: $
 *
 */

#ifndef   	PYREGFI_H_
# define   	PYREGFI_H_

#include "class.h"
#include "aff4_errors.h"
#include "regfi.h"

/** Forward declarations */
struct RegistryFile_t;
struct RegistryKey_t;
struct SubkeyIterator_t;
struct ValueIterator_t;
struct TreeIterator_t;

BIND_STRUCT(REGFI_NK)
BIND_STRUCT(REGFI_VK)
BIND_STRUCT(REGFI_DATA)

/** This is the base class for data objects */
CLASS(RawData, Object)
    const REGFI_DATA *data;
    const REGFI_VK *rec;

    RawData METHOD(RawData, Con, REGFI_DATA *data, REGFI_VK *record);

    /** Return the raw buffer as a string. By default we only return
        this much data - specify a required length to return more.

        DEFAULT(len) = 4096;
    */
    int METHOD(RawData, get_value, OUT char *buffer, int len);
END_CLASS

CLASS(DataString, RawData)
     BORROWED char *METHOD(DataString, get_value);
END_CLASS

CLASS(DWORDData, RawData)
     uint64_t METHOD(DWORDData, get_value);
END_CLASS

/** This is an iterator for traversing an entire registry hive */
CLASS(TreeIterator, Object)
     PRIVATE REGFI_ITERATOR *iter;
     PRIVATE struct RegistryFile_t *file;
     PRIVATE bool root_traversed;

     struct TreeIterator_t *METHOD(TreeIterator, Con, struct RegistryFile_t *file,
				   char **path, REGFI_ENCODING encoding);

     void METHOD(TreeIterator, __iter__);
     struct RegistryKey_t *METHOD(TreeIterator, iternext);


     int METHOD(TreeIterator, down);
     int METHOD(TreeIterator, up);

     struct RegistryKey_t *METHOD(TreeIterator, current);
     int METHOD(TreeIterator, to_root);

END_CLASS


/** XXX */
CLASS(RegistryKey, Object)
     struct RegistryFile_t *file;
     const REGFI_NK *key;

     struct RegistryKey_t *METHOD(RegistryKey, Con,
				  struct RegistryFile_t *file, REGFI_NK *base_key);

     struct SubkeyIterator_t *METHOD(RegistryKey, subkeys);
     struct ValueIterator_t *METHOD(RegistryKey, values);

END_CLASS


/** This is an iterator for reading keys from the registry */
CLASS(SubkeyIterator, Object)
     struct RegistryFile_t *file;
     PRIVATE const REGFI_SUBKEY_LIST *list;
     PRIVATE uint32_t cur;
     
     SubkeyIterator METHOD(SubkeyIterator, Con, 
			   struct RegistryFile_t *file, REGFI_NK *key);

     void METHOD(SubkeyIterator, __iter__);
     RegistryKey METHOD(SubkeyIterator, iternext);
END_CLASS



/** This is an iterator for reading values from the registry */
CLASS(ValueIterator, Object)
     struct RegistryFile_t *file;
     PRIVATE const REGFI_VALUE_LIST *list;
     PRIVATE uint32_t cur;
     
     ValueIterator METHOD(ValueIterator, Con, 
			  struct RegistryFile_t *file, REGFI_NK *key);

     void METHOD(ValueIterator, __iter__);
     REGFI_VK *METHOD(ValueIterator, iternext);
END_CLASS



CLASS(RegistryFile, Object)
  REGFI_FILE *reg;
  int fd;

  RegistryFile METHOD(RegistryFile, Con, char *filename);

  /* Get an iterator for a specific path in the register if path is
     specified.

     XXX: can we switch to UTF-8 and have Python properly import that?

     DEFAULT(path) == NULL;
     DEFAULT(encoding) = REGFI_ENCODING_ASCII;
  */
  TreeIterator METHOD(RegistryFile, TreeIterator, char **path, REGFI_ENCODING encoding);

  /** Set the verbosity level of messages generated by the library for the 
 *  current thread.
 *
 * @param mask   An integer representing the types of messages desired.
 *               Acceptable values are created through bitwise ORs of 
 *               REGFI_LOG_* values.  For instance, if only errors and
 *               informational messages were desired (but not warnings),
 *               then one would specify: REGFI_LOG_ERROR|REGFI_LOG_INFO
 *               By default the message mask is: REGFI_LOG_ERROR|REGFI_LOG_WARN.
 *
 * @return       true on success and false on failure.  Failure occurs if 
 *               underlying pthread functions fail.  errno is set in this case.
 *
 * Message masks are set in a thread-specific way.  If one were to set a message
 * mask in one thread and then spawn a new thread, then the new thread will have
 * it's message mask reset to the default.  This function may be called at any 
 * time and will take effect immediately for the current thread.
 *
 * @note When a non-zero message mask is set, messages will
 *       accumulate in memory without limit if they are not fetched using
 *       @ref regfi_get_log_str and subsequently freed by the caller.  It is
 *       recommended that messsages be fetched after each regfi API call in
 *       order to provide the most context.
 *
 * @ingroup regfiBase
 */
  int METHOD(RegistryFile, set_log_mask, uint16_t mask);

  
END_CLASS


void pyregfi_init();

#endif 	    /* !PYREGFI_H_ */
